/**
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2017, ADIT GmbH
 *
 * This file is part of GENIVI Project AudioManager.
 *
 * Contributions are licensed to the GENIVI Alliance under one or more
 * Contribution License Agreements.
 *
 * \copyright
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License, v. 2.0. If a  copy of the MPL was not distributed with
 * this file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * \author Jens Lorenz, jlorenz@de.adit-jv.com ADIT 2017
 * \author Mattia Guerra, mguerra@de.adit-jv.com ADIT 2017
 *
 * \file CAmLoggerFile.cpp
 * For further information see http://www.genivi.org/.
 *
 */

#include <algorithm>
#include <iostream>
#include <sstream>
#include <mutex>
#include <iomanip>
#include "CAmLogWrapper.h"
#include "CAmLoggerFile.h"
#include "CAmTimeUtility.h"

using namespace std;

namespace am
{

#define PADDING_WIDTH 4
#define PADDING_WAY setw(PADDING_WIDTH)
#define PADDING_ALIGN left

pthread_mutex_t gFileMtx = PTHREAD_MUTEX_INITIALIZER;

CAmLogContextFile::CAmLogContextFile(const char* id, const am_LogLevel_e level, std::ofstream *Filestream) :
    mId(id), mLogLevel(level), mpFilestream(Filestream)
{
}

void CAmLogContextFile::append(const int8_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const uint8_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const int16_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const uint16_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const int32_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const uint32_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const uint64_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const int64_t value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const bool value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const char * value)
{
    appendFile(value);
}

void CAmLogContextFile::append(const vector<uint8_t> & data)
{
    mBuffer << data.data() << " ";
}

template<class T> void CAmLogContextFile::appendFile(T value)
{
    mBuffer << value << " ";
}

bool CAmLogContextFile::configure(const am_LogLevel_e loglevel)
{
    if (loglevel > mLogLevel)
        return false;

    pthread_mutex_lock(&gFileMtx);
    (*mpFilestream) << CAmTimeUtility::now() << "[" << PADDING_WAY << PADDING_ALIGN << string(mId, 0 , PADDING_WIDTH) << "] ";

    return true;
}

bool CAmLogContextFile::checkLogLevel(const am_LogLevel_e logLevel)
{
    return logLevel <= mLogLevel;
}

void CAmLogContextFile::send()
{
    (*mpFilestream) << mBuffer.str().c_str() << endl;
    mBuffer.str("");
    mBuffer.clear();
    pthread_mutex_unlock(&gFileMtx);
}



CAmLoggerFile::CAmLoggerFile(const bool debugOn, const string & filename, const bool onlyError) : mDebugOn(debugOn), mStandardLogLevel(LL_INFO)
{
    if (onlyError)
    {
        mStandardLogLevel = LL_ERROR;
    }
    mFilestream.open(filename.c_str(), ofstream::out | ofstream::trunc);
    if (!mFilestream.is_open())
    {
            throw runtime_error("Cannot open file for logging");
    }
    if (!mDebugOn)
    {
        print("Running without Logging support");
        return;
    }
}

CAmLoggerFile::~CAmLoggerFile()
{
    mFilestream.close();
    unregisterApp();
}

void CAmLoggerFile::unregisterApp()
{
    for (auto&& context : mCtxTable)
        unregisterContext(context.first);
}

void CAmLoggerFile::registerApp(const char *appid, const char * description)

{
    print("Register Application " + string(appid, PADDING_WIDTH) + ", " + string(description));
    registerContext(DEFAULT_CONTEXT, DEFAULT_DESCRIPTION);
}

IAmLogContext& CAmLoggerFile::registerContext(const char * contextid, const char * description)
{
    return registerContext(contextid, description, mStandardLogLevel, (mDebugOn) ? LS_ON : LS_OFF);
}

IAmLogContext& CAmLoggerFile::registerContext(const char * contextid, const char * description,
        const am_LogLevel_e level, const am_LogStatus_e status)
{
    auto&& entry = mCtxTable.find(contextid);
    if (entry == mCtxTable.end())
    {
        std::pair<std::map<const char*, CAmLogContextFile*>::iterator, bool> result = mCtxTable.insert(std::make_pair(contextid, new CAmLogContextFile(contextid, level, &mFilestream)));
        entry = result.first;
    }

    if (mDebugOn)
        print("Registering Context " + string(contextid, PADDING_WIDTH) + ", " + description);

    return *entry->second;
}

IAmLogContext& CAmLoggerFile::importContext(const char* contextid)
{
    return *(mCtxTable[contextid ? contextid : DEFAULT_CONTEXT]);
}

void CAmLoggerFile::unregisterContext(const char* contextid)
{
    delete mCtxTable[contextid];
    mCtxTable.erase(contextid);
    print("Context " + string(contextid, PADDING_WIDTH) + "unregistered");
}

void CAmLoggerFile::print(string str)
{
    mFilestream << CAmTimeUtility::now() << "[" << PADDING_WAY << PADDING_ALIGN << "LOG" << "] " << str << endl;
}
}
